/*******************************************************************************
 * Copyright (c) 2000, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.swt.widgets;

import org.eclipse.swt.*;
import org.eclipse.swt.graphics.*;

public class TabItem extends Item {
	TabFolder parent;
	Control control;
	String toolTipText;

public TabItem (TabFolder parent, int style) {
	this (parent, style, -1);
}

public TabItem (TabFolder parent, int style, int index) {
	super (parent, style, index);
}

protected void checkSubclass () {
	if (!isValidSubclass ()) error (SWT.ERROR_INVALID_SUBCLASS);
}

void createWidget(Display display, Widget parent, int style, int index) {
	this.display = display;
	this.parent = (TabFolder) parent;
	int newIndex = index;
	if (index==-1) { newIndex = this.parent.getItemCount();}
	this.parent.createItem (this, newIndex);
	hookEvents ();
	register ();
	text = "";
}

void destroyWidget() {
	super.destroyWidget();
	parent.removeItem (this);
}

public Control getControl () {
	checkWidget();
	return control;
}

public TabFolder getParent () {
	checkWidget();
	return parent;
}

public String getToolTipText () {
	checkWidget();
	return toolTipText;
}

void hookEvents () {
	_hookEvents("onmousedown", 	SWT.MouseDown);
	_hookEvents("onfocus", 		SWT.FocusIn);
	_hookEvents("onblur", 		SWT.FocusOut);
}

void releaseHandle () {
	super.releaseHandle ();
	parent = null;
}

void releaseParent () {
	super.releaseParent ();
	int index = parent.indexOf (this);
	if (index == parent.getSelectionIndex ()) {
		if (control != null) control.setVisible (false);
	}
}

void releaseWidget () {
	super.releaseWidget ();
	control = null;
}

public void setControl (Control control) {
	checkWidget();
	if (control != null) {
		if (control.isDisposed()) error (SWT.ERROR_INVALID_ARGUMENT);
		if (control.parent != parent) error (SWT.ERROR_INVALID_PARENT);
	}
	if (this.control != null && this.control.isDisposed ()) {
		this.control = null;
	}
	Control oldControl = this.control, newControl = control;
	this.control = control;
	_setControl (control);
	int index = parent.indexOf (this);
	if (index != parent.getSelectionIndex ()) {
		if (newControl != null) newControl.setVisible (false);
		return;
	}
	if (newControl != null) {
		newControl.setBounds (parent.getClientArea ());
		newControl.setVisible (true);
	}
	if (oldControl != null) oldControl.setVisible (false);
}

public void setImage (Image image) {
	checkWidget();
	int index = parent.indexOf (this);
	if (index == -1) return;
	super.setImage (image);
	_setImage (image);
}

public void setText (String string) {
	checkWidget();
	if (string == null) error (SWT.ERROR_NULL_ARGUMENT);
	if (string.equals (text)) return;
	int index = parent.indexOf (this);
	if (index == -1) return;
	super.setText (string);
	_setText (string);
}

public void setToolTipText (String string) {
	checkWidget();
	toolTipText = string;
	_setToolTipText (string);
}

/*---------------- NATIVE INTERFACE ----------------*/

protected native void _createItem (Widget parent, int style, int index) /*-{
	if(!$wnd.dojo._hasResource["org.eclipse.swt.TabItem"]){
		$wnd.dojo._hasResource["org.eclipse.swt.TabItem"] = true;
		$wnd.dojo.provide("org.eclipse.swt.TabItem");
		
		$wnd.dojo.require("dijit._Widget");
	
		$wnd.dojo.declare("org.eclipse.swt.TabItem", $wnd.dijit._Widget, {
			index: 0,
			setControl: function(control){
				this.domNode.appendChild(control.domNode);
				$wnd.dojo.publish("org.eclipse.swt.TabItem::setControl",[this]);
			}
		});
	}	
	var jsParent = parent.@org.eclipse.swt.widgets.Widget::jsObject;
	var params = {};
	params.index = index;
	var self = new $wnd.org.eclipse.swt.TabItem(params);
	try {
		this.@org.eclipse.swt.widgets.Widget::jsObject = self;
		jsParent.addChild(self, index);
	} catch (e) {
//TODO Have to throw real exception for Java side also	
		$wnd.console.log(e);
	}
}-*/;

native void _setControl (Control control) /*-{
	this.@org.eclipse.swt.widgets.Widget::jsObject.setControl(control.@org.eclipse.swt.widgets.Widget::jsObject);	
}-*/;

void _setImage (Image image) {
	callMethod("_setImage", null, new Object[]{image});
}

native void _setText (String string) /*-{
	this.@org.eclipse.swt.widgets.Widget::jsObject.title = string
}-*/;

void _setToolTipText (String string) {
	callMethod ("_setToolTipText", null, new Object[]{string});
}

}
